/*
 * Decompiled with CFR 0.152.
 */
package be.iminds.ilabt.jfed.util.lib;

import be.iminds.ilabt.jfed.util.library.KeyUtil;
import be.iminds.ilabt.jfed.util.library.XmlUtil;
import java.io.IOException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import javax.xml.transform.TransformerException;
import org.apache.xml.security.Init;
import org.apache.xml.security.exceptions.XMLSecurityException;
import org.apache.xml.security.keys.KeyInfo;
import org.apache.xml.security.keys.storage.StorageResolver;
import org.apache.xml.security.keys.storage.StorageResolverSpi;
import org.apache.xml.security.keys.storage.implementations.KeyStoreResolver;
import org.apache.xml.security.signature.XMLSignature;
import org.apache.xml.security.utils.ElementProxy;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.Assert;
import org.testng.annotations.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class XmlUtilTest {
    private static final Logger LOG = LoggerFactory.getLogger(XmlUtilTest.class);
    private static final String testPemPrivateKey = "-----BEGIN RSA PRIVATE KEY-----\nMIIEowIBAAKCAQEAt9DLTl6GDWg7RgCzepWyV9EpUfXUZUqQi1D8kJAWR5IcnyR1\nwCt5BqfUR5uBtvn2m2XSz3nD0jT6+sGsx0G1+50h31ukOA1JnRlvgENNiJztio76\nUP9eN0IrtFpuBK+g50kRrlrUhG/zfAV8snQpBO1Bue4YLJTf9mmnusTL2gWrzglG\nMalOfbIU2H3yjfPh5es5VOVutvNXpGGKVYoJrlOZ1SgkKwhlt+ANyrj6+cERaVqb\ngq1TeOOsFjdEq59HmP74eHTu0v9WrrWrlRbYElENbhDhGL2YXf0H9B+0eliTbkOl\ny8ACIhGurjLKiPWIs6uKpaLLxcF0teQwsg0IFQIDAQABAoIBABacqsXjPt5bWh9M\nasfDroK7Cjg4433qZEmfFtK5xuPbB4ZUN+AB+QtmjeiZKqXVrg6DwiEfu4UaW12J\ngbWpUbe4JriRT2xNxiZ7QIoovVr24wnOeOCJ6Y8mprWzAiG1cTfCd1hZeBcqpms1\nSZeyTnF6tkEnFvY1AjAP40BIgUM6hamTNPZgwRuob2ISHrLljm1AIOGRTY1hDISi\nKpfyY7yHTZJLwfUQVD6KBLg9YeiGy1NPzE9UmXrQlRooiipLw/kGA6SAoUblqvS3\nkFqTg/qUFxF4VxbCg4FjGXCQY4ws71OfGo4zgN/DmiO1vrHKnJvLIkjpTacMy2gr\nPrhWWUECgYEA8pX9euSto87peYxYkDYnQXYAK65X4mFIOV8CYW6hXMbltS5z/1hW\ndZKCNJr8ui6LpbcWTKmTYjDSxR15w+kuF1Trib682pQ2E3LCPYneBOJ0Hogv+D02\nKF5PsCOYcReVwHq8H1dEck9j2g5wGLaajDWw3Vu7o3trT6zbUQD72YkCgYEAwfrc\nRfKOk53HLSipM7Ipr+hPtYLrUExQTip8dh+3mAtXeMOe/Nm8P5ERSBAa8ETNx9fD                                                                                                                              \npzcQY/7GqgtLwAzgIM73yRSnUe0mDV4JwCeteJ8wpdB+xbPprPgp+4xj0oO0zfrh                                                                                                                              \n9Gi7Wl9lsFHGz03WLcLks2JNk9yEbWdStIUFsy0CgYEA1wkaLwvy2BgcYkCEaSF3                                                                                                                              \n3tEpCY98HBI8IY8U3yVgKIuvJ9JLxPebfJEollUk3i1QYM9RFxfiq9G3pD+ipoVW                                                                                                                              \n9Bu0xt+rLN7gqJC7mi1R/2Z1pHL4gsZP8azYWrzS7r0HZB3oEKgc5m91hd/yghWs                                                                                                                              \nHcmiRDaCuEI5N+F0b/O9THECgYAPqqARsiZbu7lzZz5LI4+EWoHdigZRROiTd5D3                                                                                                                              \nxVYBnKDi6wGzMC6o/6FirZJPZ28rPbzai3LycIhBAYqLpz6Ox8hxYA6najzRhY/K                                                                                                                              \nCipTfJ/kNHOYqhAsXRqe61bAkMyQeju3jqHbCatf0yr4YTP0HDJ2XnEfiQbVAyxF                                                                                                                              \n0hyEvQKBgDOyK9omGnMAFTZOMUu94y+atccMqHyxCDLZl6yh9ipB+gvGTDh9c6+G\nmMua/QVmH8yFT3NWM++2P54vuUFgrn3WmIpq4pkIoQ5qF9xi192oC7Kocwwy3OTn\nlmyfwPhutAcGcMFZcQoNPXPzrCj1Ku4duXtO6jkWy3kTqvr8ib8j\n-----END RSA PRIVATE KEY-----";
    private static final String testPemCert = "-----BEGIN CERTIFICATE-----\nMIIEozCCA4ugAwIBAgIJAKdm6CkA/TLMMA0GCSqGSIb3DQEBBQUAMIGRMQswCQYD\nVQQGEwJCRTETMBEGA1UECBMKU29tZS1TdGF0ZTEOMAwGA1UEBxMFR2hlbnQxDzAN\nBgNVBAoTBmlNaW5kczEPMA0GA1UECxMGaWxhYi50MRQwEgYDVQQDEwtqRmVkIFRl\nc3RlcjElMCMGCSqGSIb3DQEJARYWamZlZEBsaXN0cy5mZWQ0ZmlyZS5ldTAgFw0x\nNDAzMjYwNzM1MThaGA8yMTUxMDIxNjA3MzUxOFowgZExCzAJBgNVBAYTAkJFMRMw\nEQYDVQQIEwpTb21lLVN0YXRlMQ4wDAYDVQQHEwVHaGVudDEPMA0GA1UEChMGaU1p\nbmRzMQ8wDQYDVQQLEwZpbGFiLnQxFDASBgNVBAMTC2pGZWQgVGVzdGVyMSUwIwYJ\nKoZIhvcNAQkBFhZqZmVkQGxpc3RzLmZlZDRmaXJlLmV1MIIBIjANBgkqhkiG9w0B\nAQEFAAOCAQ8AMIIBCgKCAQEAt9DLTl6GDWg7RgCzepWyV9EpUfXUZUqQi1D8kJAW\nR5IcnyR1wCt5BqfUR5uBtvn2m2XSz3nD0jT6+sGsx0G1+50h31ukOA1JnRlvgENN\niJztio76UP9eN0IrtFpuBK+g50kRrlrUhG/zfAV8snQpBO1Bue4YLJTf9mmnusTL\n2gWrzglGMalOfbIU2H3yjfPh5es5VOVutvNXpGGKVYoJrlOZ1SgkKwhlt+ANyrj6\n+cERaVqbgq1TeOOsFjdEq59HmP74eHTu0v9WrrWrlRbYElENbhDhGL2YXf0H9B+0\neliTbkOly8ACIhGurjLKiPWIs6uKpaLLxcF0teQwsg0IFQIDAQABo4H5MIH2MB0G\nA1UdDgQWBBSN6TiaD3UUm9aRBzS7fACP/+BZBDCBxgYDVR0jBIG+MIG7gBSN6Tia\nD3UUm9aRBzS7fACP/+BZBKGBl6SBlDCBkTELMAkGA1UEBhMCQkUxEzARBgNVBAgT\nClNvbWUtU3RhdGUxDjAMBgNVBAcTBUdoZW50MQ8wDQYDVQQKEwZpTWluZHMxDzAN\nBgNVBAsTBmlsYWIudDEUMBIGA1UEAxMLakZlZCBUZXN0ZXIxJTAjBgkqhkiG9w0B\nCQEWFmpmZWRAbGlzdHMuZmVkNGZpcmUuZXWCCQCnZugpAP0yzDAMBgNVHRMEBTAD\nAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQA/QKeMMXDvoWzPYmPhnQKro4Bk7YKd9HGk\nrwQKAx2Qs/LLXV3pp6vYJ9t66XisfPZlz5l5L5uDzTdU98ugvrfdxCCPkPixFzty\ntS1hbQXSc4V4Zv1tMy6cyKJOhzGR3t8f2tgn4kiBd6z1PaYci92f00/HuI8JQEYb\nOzC63ZboIHyK14voKVomqXANqVlm+UdSjt8L+v5v1vxt/hIeDZQHwHcP9iTvuNT1\nHZ47S38JUdX1vHU+YAMHetuBpDGOsM54OJT7IlXD140WNbgznydvhEDK9cZZhb7D\nHJ/pNDUIkpUrhxlovpFi9rdhShOYL6XJEq4MlhDRNTM19qGRZIo9\n-----END CERTIFICATE-----";
    private static final PrivateKey testPrivateKey;
    private static final PublicKey testPublicKey;
    private static final X509Certificate testCert;
    private final String simple_xml1 = "<testxml><t>test</t><tt/></testxml>";
    private final String simple_xml2 = "<testxml/>";
    private final String simple_xml3 = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<testxml><t>test</t><tt/></testxml>\n";
    private final String simple_xml4 = "<testxml>blablabla<bluh/>bla\nblabla</testxml>\n";
    private final String simple_xml5 = "<testxml><t>test</t><tt/></testxml>";
    private final String simple_xml6 = "<note><to>You</to><from>Me</from><subject>msg</subject><body>This is a test!</body></note>";
    private final String simple_xml7 = "<testxml att=\"blah\" y=\"blih\"><t>test</t><tt att=\"test\"/></testxml>";
    private final String simple_xml8 = "<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>";
    private final String bad_xml1 = "blah<testxml><t>test</t><tt/></testxml>";
    private final String bad_xml2 = "<testxml <oops><t>test</t><tt/></testxml>";
    private final String bad_xml3 = "<testxml><t>test</t></><tt/></testxml>";
    private final String bad_xml4 = "<testxml><badnamespace:t>test</badnamespace:t></><tt/></testxml>";
    private final String bad_xml5 = "<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></testxml>";
    private final String bad_xml6 = "<testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>";
    private final String bad_xml7 = "<testxml badatt=bad><t>test</t></><tt/></testxml>";
    private final String bad_xml8 = "<testxml><t>test</t></><tt badatt=bad/></testxml>";
    private final String bad_xml9 = "<testxml><t>test</t badatt=\"bad\"></><tt/></testxml>";
    private final String bad_xml10 = "<testxml><t>test</T></><tt/></testxml>";
    private final String bad_xml11 = "<testxml><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><t>test</T></><tt/></testxml>";
    private final String java_signed_result = "<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"no\"?><testxml>   <test xml:id=\"id0\">      <name>test</name>      <foo>foo</foo>   </test>   <signatures><Signature xmlns=\"http://www.w3.org/2000/09/xmldsig#\" xml:id=\"Sig_id0\">\n<SignedInfo>\n<CanonicalizationMethod Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n<SignatureMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#rsa-sha1\"/>\n<Reference URI=\"#id0\">\n<Transforms>\n<Transform Algorithm=\"http://www.w3.org/2000/09/xmldsig#enveloped-signature\"/>\n<Transform Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n</Transforms>\n<DigestMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#sha1\"/>\n<DigestValue>UiSQzKxdyW3WsDIlgxcaBD9K6UU=</DigestValue>\n</Reference>\n</SignedInfo>\n<SignatureValue>qAJyoT7hWTZUeVgSn47S39ZGczdjHzIW9t0bHnzPSHbkKUXQg/UO/xKYQHUrKYPYYd7jm+HDIWRX\nElOxgD6kT/j16TAYBPq/iVVkJXnbLxn1++A9q82iH7Dg5Bjkh1jwCf9nWmjCKsIgyib0+yEHu89O\nCJOFAa2Jsc6RFJyQ3bcK7tKIF+YJQ/3IIpUnZWNxM1zBRmZUNx82bImTixU4ILUB7biJBQZiJHCd\nbns9F3c8KaCHxl/6xtqUppr6RPj6qhJF56zQxbcYPsMVkTjTL7juj6TWwAQIc2QN+MkmfNDawVN5\n2BZum6lhLSTdVrwcOYyfxUDGXNRiLjE8Tw/n0w==</SignatureValue>\n<KeyInfo>\n<X509Data>\n<X509Certificate>MIIEozCCA4ugAwIBAgIJAKdm6CkA/TLMMA0GCSqGSIb3DQEBBQUAMIGRMQswCQYDVQQGEwJCRTET\nMBEGA1UECBMKU29tZS1TdGF0ZTEOMAwGA1UEBxMFR2hlbnQxDzANBgNVBAoTBmlNaW5kczEPMA0G\nA1UECxMGaWxhYi50MRQwEgYDVQQDEwtqRmVkIFRlc3RlcjElMCMGCSqGSIb3DQEJARYWamZlZEBs\naXN0cy5mZWQ0ZmlyZS5ldTAgFw0xNDAzMjYwNzM1MThaGA8yMTUxMDIxNjA3MzUxOFowgZExCzAJ\nBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMQ4wDAYDVQQHEwVHaGVudDEPMA0GA1UEChMG\naU1pbmRzMQ8wDQYDVQQLEwZpbGFiLnQxFDASBgNVBAMTC2pGZWQgVGVzdGVyMSUwIwYJKoZIhvcN\nAQkBFhZqZmVkQGxpc3RzLmZlZDRmaXJlLmV1MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\nAQEAt9DLTl6GDWg7RgCzepWyV9EpUfXUZUqQi1D8kJAWR5IcnyR1wCt5BqfUR5uBtvn2m2XSz3nD\n0jT6+sGsx0G1+50h31ukOA1JnRlvgENNiJztio76UP9eN0IrtFpuBK+g50kRrlrUhG/zfAV8snQp\nBO1Bue4YLJTf9mmnusTL2gWrzglGMalOfbIU2H3yjfPh5es5VOVutvNXpGGKVYoJrlOZ1SgkKwhl\nt+ANyrj6+cERaVqbgq1TeOOsFjdEq59HmP74eHTu0v9WrrWrlRbYElENbhDhGL2YXf0H9B+0eliT\nbkOly8ACIhGurjLKiPWIs6uKpaLLxcF0teQwsg0IFQIDAQABo4H5MIH2MB0GA1UdDgQWBBSN6Tia\nD3UUm9aRBzS7fACP/+BZBDCBxgYDVR0jBIG+MIG7gBSN6TiaD3UUm9aRBzS7fACP/+BZBKGBl6SB\nlDCBkTELMAkGA1UEBhMCQkUxEzARBgNVBAgTClNvbWUtU3RhdGUxDjAMBgNVBAcTBUdoZW50MQ8w\nDQYDVQQKEwZpTWluZHMxDzANBgNVBAsTBmlsYWIudDEUMBIGA1UEAxMLakZlZCBUZXN0ZXIxJTAj\nBgkqhkiG9w0BCQEWFmpmZWRAbGlzdHMuZmVkNGZpcmUuZXWCCQCnZugpAP0yzDAMBgNVHRMEBTAD\nAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQA/QKeMMXDvoWzPYmPhnQKro4Bk7YKd9HGkrwQKAx2Qs/LL\nXV3pp6vYJ9t66XisfPZlz5l5L5uDzTdU98ugvrfdxCCPkPixFztytS1hbQXSc4V4Zv1tMy6cyKJO\nhzGR3t8f2tgn4kiBd6z1PaYci92f00/HuI8JQEYbOzC63ZboIHyK14voKVomqXANqVlm+UdSjt8L\n+v5v1vxt/hIeDZQHwHcP9iTvuNT1HZ47S38JUdX1vHU+YAMHetuBpDGOsM54OJT7IlXD140WNbgz\nnydvhEDK9cZZhb7DHJ/pNDUIkpUrhxlovpFi9rdhShOYL6XJEq4MlhDRNTM19qGRZIo9</X509Certificate>\n</X509Data>\n</KeyInfo>\n</Signature></signatures></testxml>";
    private final String xmlsec1_signed_result = "<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"no\"?>\n<testxml>\n   <test xml:id=\"id0\">\n      <name>test</name>\n      <foo>foo</foo>\n   </test>\n   <signatures>\n<Signature xmlns=\"http://www.w3.org/2000/09/xmldsig#\" xml:id=\"Sig_id0\">\n<SignedInfo>\n<CanonicalizationMethod Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n<SignatureMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#rsa-sha1\"/>\n<Reference>\n<Transforms>\n<Transform Algorithm=\"http://www.w3.org/2000/09/xmldsig#enveloped-signature\"/>\n<Transform Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n</Transforms>\n<DigestMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#sha1\"/>\n<DigestValue>8yKbnyLb03JAVQJbiRw09uo/Ads=</DigestValue>\n</Reference>\n</SignedInfo>\n<SignatureValue>TF547ZV8Yk1nse7r7GUoM0Ys8Hjr5zib/J47Tw8ApEZ2K57IbnzrGajiOXnTW/mI\nXfdelmHVm4FDaIUuroD2L6TDKOkfjBiYeYOZ97+VVwOwk30aPRnobdYU39oflcsR\nKeU4Pzc5delWDfqn/3Z7kaEW2WoMLtbOa0vEhwBts95lc1ODSX2b6f6iKrboen9f\n8LdRblS+LTPvFeQ6NfEuFwBfLI5OG0LmM6Fk0/WU6m/P7OohUm3/7HUyrzzLD4OD\numb/e6UmadG1m8vxP91PwleLRph/biHAxNF12Zx3l0ZdIh9PEprEcC5H2gh5KuX4\nJdIQ0x2Nm9eg6OxNcoCJnQ==</SignatureValue>\n<KeyInfo>\n<X509Data>\n<X509Certificate>MIIEozCCA4ugAwIBAgIJAKdm6CkA/TLMMA0GCSqGSIb3DQEBBQUAMIGRMQswCQYD\nVQQGEwJCRTETMBEGA1UECBMKU29tZS1TdGF0ZTEOMAwGA1UEBxMFR2hlbnQxDzAN\nBgNVBAoTBmlNaW5kczEPMA0GA1UECxMGaWxhYi50MRQwEgYDVQQDEwtqRmVkIFRl\nc3RlcjElMCMGCSqGSIb3DQEJARYWamZlZEBsaXN0cy5mZWQ0ZmlyZS5ldTAgFw0x\nNDAzMjYwNzM1MThaGA8yMTUxMDIxNjA3MzUxOFowgZExCzAJBgNVBAYTAkJFMRMw\nEQYDVQQIEwpTb21lLVN0YXRlMQ4wDAYDVQQHEwVHaGVudDEPMA0GA1UEChMGaU1p\nbmRzMQ8wDQYDVQQLEwZpbGFiLnQxFDASBgNVBAMTC2pGZWQgVGVzdGVyMSUwIwYJ\nKoZIhvcNAQkBFhZqZmVkQGxpc3RzLmZlZDRmaXJlLmV1MIIBIjANBgkqhkiG9w0B\nAQEFAAOCAQ8AMIIBCgKCAQEAt9DLTl6GDWg7RgCzepWyV9EpUfXUZUqQi1D8kJAW\nR5IcnyR1wCt5BqfUR5uBtvn2m2XSz3nD0jT6+sGsx0G1+50h31ukOA1JnRlvgENN\niJztio76UP9eN0IrtFpuBK+g50kRrlrUhG/zfAV8snQpBO1Bue4YLJTf9mmnusTL\n2gWrzglGMalOfbIU2H3yjfPh5es5VOVutvNXpGGKVYoJrlOZ1SgkKwhlt+ANyrj6\n+cERaVqbgq1TeOOsFjdEq59HmP74eHTu0v9WrrWrlRbYElENbhDhGL2YXf0H9B+0\neliTbkOly8ACIhGurjLKiPWIs6uKpaLLxcF0teQwsg0IFQIDAQABo4H5MIH2MB0G\nA1UdDgQWBBSN6TiaD3UUm9aRBzS7fACP/+BZBDCBxgYDVR0jBIG+MIG7gBSN6Tia\nD3UUm9aRBzS7fACP/+BZBKGBl6SBlDCBkTELMAkGA1UEBhMCQkUxEzARBgNVBAgT\nClNvbWUtU3RhdGUxDjAMBgNVBAcTBUdoZW50MQ8wDQYDVQQKEwZpTWluZHMxDzAN\nBgNVBAsTBmlsYWIudDEUMBIGA1UEAxMLakZlZCBUZXN0ZXIxJTAjBgkqhkiG9w0B\nCQEWFmpmZWRAbGlzdHMuZmVkNGZpcmUuZXWCCQCnZugpAP0yzDAMBgNVHRMEBTAD\nAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQA/QKeMMXDvoWzPYmPhnQKro4Bk7YKd9HGk\nrwQKAx2Qs/LLXV3pp6vYJ9t66XisfPZlz5l5L5uDzTdU98ugvrfdxCCPkPixFzty\ntS1hbQXSc4V4Zv1tMy6cyKJOhzGR3t8f2tgn4kiBd6z1PaYci92f00/HuI8JQEYb\nOzC63ZboIHyK14voKVomqXANqVlm+UdSjt8L+v5v1vxt/hIeDZQHwHcP9iTvuNT1\nHZ47S38JUdX1vHU+YAMHetuBpDGOsM54OJT7IlXD140WNbgznydvhEDK9cZZhb7D\nHJ/pNDUIkpUrhxlovpFi9rdhShOYL6XJEq4MlhDRNTM19qGRZIo9</X509Certificate>\n</X509Data>\n</KeyInfo>\n</Signature>\n   </signatures>\n</testxml>";
    private static final String unformattedXml = "<test format=\"notyet\"><inner ok=\"ok\">blah</inner><inner nr=\"2\">3</inner><yup/></test>";
    private static final String badunformattedXml = "<test format=\"notyet\"><inner ok=\"ok\">blah</inner><inner nr=\"2\">3</inner><yup/>oops";

    private boolean checkSig(String xmlString) throws CertificateException, NoSuchAlgorithmException, KeyStoreException, IOException {
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)xmlString, (boolean)true);
        XmlUtil.markXmlIdsAsXmlId((Document)doc);
        return this.checkSig(doc);
    }

    private boolean checkSig(Document xmlDoc) throws KeyStoreException, CertificateException, NoSuchAlgorithmException, IOException {
        KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
        trustStore.load(null, "any_pass".toCharArray());
        trustStore.setCertificateEntry("testCert", testCert);
        assert (xmlDoc != null);
        NodeList signatureNodeList = xmlDoc.getElementsByTagNameNS("http://www.w3.org/2000/09/xmldsig#", "Signature");
        if (signatureNodeList.getLength() == 0) {
            LOG.trace("No Signature found in xml file: " + XmlUtil.elementToXmlString((Element)xmlDoc.getDocumentElement()));
            return false;
        }
        LOG.trace("There are " + signatureNodeList.getLength() + " <signature> elements in <Signature> that need to be checked.");
        try {
            for (int i = 0; i < signatureNodeList.getLength(); ++i) {
                Element sigElement = (Element)signatureNodeList.item(i);
                LOG.trace("Checking <signature> with apache santurio library");
                Init.init();
                ElementProxy.setDefaultPrefix((String)"http://www.w3.org/2000/09/xmldsig#", (String)"");
                XMLSignature signature = new XMLSignature(sigElement, null);
                KeyInfo ki = signature.getKeyInfo();
                ki.addStorageResolver(new StorageResolver((StorageResolverSpi)new KeyStoreResolver(trustStore)));
                if (ki != null) {
                    X509Certificate cert = signature.getKeyInfo().getX509Certificate();
                    if (cert != null) {
                        boolean check = signature.checkSignatureValue(cert);
                        LOG.trace("X509Certificate Check: " + check);
                        if (check) continue;
                        return false;
                    }
                    PublicKey pk = signature.getKeyInfo().getPublicKey();
                    if (pk != null) {
                        boolean check = signature.checkSignatureValue((Key)pk);
                        LOG.trace("PublicKey Check: " + check);
                        if (check) continue;
                        return false;
                    }
                    LOG.trace("Could not find Certificate or PublicKey");
                    return false;
                }
                LOG.trace("Could not find ds:KeyInfo");
                return false;
            }
        }
        catch (Exception e) {
            LOG.error("Error during checkSignedCredential", (Throwable)e);
            return false;
        }
        return true;
    }

    @Test
    public void testXmlSign() throws TransformerException, XMLSecurityException, CertificateException, NoSuchAlgorithmException, KeyStoreException, IOException {
        assert (testCert != null);
        assert (testPublicKey != null);
        assert (testPrivateKey != null);
        String testXml = "<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"no\"?><testxml>   <test xml:id=\"id0\">      <name>test</name>      <foo>foo</foo>   </test>   <signatures></signatures></testxml>";
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)testXml);
        XmlUtil.markXmlIdsAsXmlId((Document)doc);
        String elementToSignId = "id0";
        String elementToAddSignatureToName = "signatures";
        String signedXml = XmlUtil.signXml((Document)doc, (String)elementToSignId, (String)elementToAddSignatureToName, (X509Certificate)testCert, (Key)testPrivateKey);
        Assert.assertTrue((boolean)this.checkSig(signedXml));
        LOG.trace("testXmlSign() signedXml=\"" + signedXml + "\"");
    }

    @Test
    public void testParseXmlFromStringMakeExceptionsRuntime1() {
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)"<testxml><t>test</t><tt/></testxml>");
        Assert.assertNotNull((Object)doc.getDocumentElement());
        Assert.assertEquals((String)doc.getDocumentElement().getTagName(), (String)"testxml");
    }

    @Test
    public void testParseXmlFromStringMakeExceptionsRuntime2() {
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)"<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"no\"?><testxml><t>test</t><tt/></testxml>");
        Assert.assertNotNull((Object)doc.getDocumentElement());
        Assert.assertEquals((String)doc.getDocumentElement().getTagName(), (String)"testxml");
    }

    @Test
    public void testParseXmlFromStringMakeExceptionsRuntime3() {
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)"<?xml version=\"1.0\" encoding=\"UTF-8\"?><testxml><t>test</t><tt/></testxml>");
        Assert.assertNotNull((Object)doc.getDocumentElement());
        Assert.assertEquals((String)doc.getDocumentElement().getTagName(), (String)"testxml");
    }

    @Test
    public void testParseXmlFromStringMakeExceptionsRuntime4() {
        Document doc = XmlUtil.parseXmlFromStringMakeExceptionsRuntime((String)"<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"no\"?><testxml>   <test xml:id=\"id0\">      <name>test</name>      <foo>foo</foo>   </test>   <signatures><Signature xmlns=\"http://www.w3.org/2000/09/xmldsig#\" xml:id=\"Sig_id0\">\n<SignedInfo>\n<CanonicalizationMethod Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n<SignatureMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#rsa-sha1\"/>\n<Reference URI=\"#id0\">\n<Transforms>\n<Transform Algorithm=\"http://www.w3.org/2000/09/xmldsig#enveloped-signature\"/>\n<Transform Algorithm=\"http://www.w3.org/2001/10/xml-exc-c14n#\"/>\n</Transforms>\n<DigestMethod Algorithm=\"http://www.w3.org/2000/09/xmldsig#sha1\"/>\n<DigestValue>UiSQzKxdyW3WsDIlgxcaBD9K6UU=</DigestValue>\n</Reference>\n</SignedInfo>\n<SignatureValue>qAJyoT7hWTZUeVgSn47S39ZGczdjHzIW9t0bHnzPSHbkKUXQg/UO/xKYQHUrKYPYYd7jm+HDIWRX\nElOxgD6kT/j16TAYBPq/iVVkJXnbLxn1++A9q82iH7Dg5Bjkh1jwCf9nWmjCKsIgyib0+yEHu89O\nCJOFAa2Jsc6RFJyQ3bcK7tKIF+YJQ/3IIpUnZWNxM1zBRmZUNx82bImTixU4ILUB7biJBQZiJHCd\nbns9F3c8KaCHxl/6xtqUppr6RPj6qhJF56zQxbcYPsMVkTjTL7juj6TWwAQIc2QN+MkmfNDawVN5\n2BZum6lhLSTdVrwcOYyfxUDGXNRiLjE8Tw/n0w==</SignatureValue>\n<KeyInfo>\n<X509Data>\n<X509Certificate>MIIEozCCA4ugAwIBAgIJAKdm6CkA/TLMMA0GCSqGSIb3DQEBBQUAMIGRMQswCQYDVQQGEwJCRTET\nMBEGA1UECBMKU29tZS1TdGF0ZTEOMAwGA1UEBxMFR2hlbnQxDzANBgNVBAoTBmlNaW5kczEPMA0G\nA1UECxMGaWxhYi50MRQwEgYDVQQDEwtqRmVkIFRlc3RlcjElMCMGCSqGSIb3DQEJARYWamZlZEBs\naXN0cy5mZWQ0ZmlyZS5ldTAgFw0xNDAzMjYwNzM1MThaGA8yMTUxMDIxNjA3MzUxOFowgZExCzAJ\nBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMQ4wDAYDVQQHEwVHaGVudDEPMA0GA1UEChMG\naU1pbmRzMQ8wDQYDVQQLEwZpbGFiLnQxFDASBgNVBAMTC2pGZWQgVGVzdGVyMSUwIwYJKoZIhvcN\nAQkBFhZqZmVkQGxpc3RzLmZlZDRmaXJlLmV1MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\nAQEAt9DLTl6GDWg7RgCzepWyV9EpUfXUZUqQi1D8kJAWR5IcnyR1wCt5BqfUR5uBtvn2m2XSz3nD\n0jT6+sGsx0G1+50h31ukOA1JnRlvgENNiJztio76UP9eN0IrtFpuBK+g50kRrlrUhG/zfAV8snQp\nBO1Bue4YLJTf9mmnusTL2gWrzglGMalOfbIU2H3yjfPh5es5VOVutvNXpGGKVYoJrlOZ1SgkKwhl\nt+ANyrj6+cERaVqbgq1TeOOsFjdEq59HmP74eHTu0v9WrrWrlRbYElENbhDhGL2YXf0H9B+0eliT\nbkOly8ACIhGurjLKiPWIs6uKpaLLxcF0teQwsg0IFQIDAQABo4H5MIH2MB0GA1UdDgQWBBSN6Tia\nD3UUm9aRBzS7fACP/+BZBDCBxgYDVR0jBIG+MIG7gBSN6TiaD3UUm9aRBzS7fACP/+BZBKGBl6SB\nlDCBkTELMAkGA1UEBhMCQkUxEzARBgNVBAgTClNvbWUtU3RhdGUxDjAMBgNVBAcTBUdoZW50MQ8w\nDQYDVQQKEwZpTWluZHMxDzANBgNVBAsTBmlsYWIudDEUMBIGA1UEAxMLakZlZCBUZXN0ZXIxJTAj\nBgkqhkiG9w0BCQEWFmpmZWRAbGlzdHMuZmVkNGZpcmUuZXWCCQCnZugpAP0yzDAMBgNVHRMEBTAD\nAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQA/QKeMMXDvoWzPYmPhnQKro4Bk7YKd9HGkrwQKAx2Qs/LL\nXV3pp6vYJ9t66XisfPZlz5l5L5uDzTdU98ugvrfdxCCPkPixFztytS1hbQXSc4V4Zv1tMy6cyKJO\nhzGR3t8f2tgn4kiBd6z1PaYci92f00/HuI8JQEYbOzC63ZboIHyK14voKVomqXANqVlm+UdSjt8L\n+v5v1vxt/hIeDZQHwHcP9iTvuNT1HZ47S38JUdX1vHU+YAMHetuBpDGOsM54OJT7IlXD140WNbgz\nnydvhEDK9cZZhb7DHJ/pNDUIkpUrhxlovpFi9rdhShOYL6XJEq4MlhDRNTM19qGRZIo9</X509Certificate>\n</X509Data>\n</KeyInfo>\n</Signature></signatures></testxml>");
        Assert.assertNotNull((Object)doc.getDocumentElement());
        Assert.assertEquals((String)doc.getDocumentElement().getTagName(), (String)"testxml");
    }

    @Test
    public void testCouldBeXmlVeryQuickTest() {
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<testxml/>"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<testxml><t>test</t><tt/></testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<testxml>blablabla<bluh/>bla\nblabla</testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<note><to>You</to><from>Me</from><subject>msg</subject><body>This is a test!</body></note>"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<testxml att=\"blah\" y=\"blih\"><t>test</t><tt att=\"test\"/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>"));
        Assert.assertFalse((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"blah<testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)"  <testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.couldBeXmlVeryQuickTest((String)" \n\t <testxml><t>test</t><tt/></testxml>"));
    }

    @Test
    public void testIsXmlLikeQuickTest() {
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml/>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<testxml><t>test</t><tt/></testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml>blablabla<bluh/>bla\nblabla</testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<note><to>You</to><from>Me</from><subject>msg</subject><body>This is a test!</body></note>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml att=\"blah\" y=\"blih\"><t>test</t><tt att=\"test\"/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"blah<testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"  <testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)" \n\t <testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml <oops><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><badnamespace:t>test</badnamespace:t></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml badatt=bad><t>test</t></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><t>test</t></><tt badatt=bad/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><t>test</t badatt=\"bad\"></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><t>test</T></><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isXmlLikeQuickTest((String)"<testxml><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><t>test</T></><tt/></testxml>"));
    }

    @Test
    public void testIsWellFormedXmlQuickTest() {
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml/>"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<testxml><t>test</t><tt/></testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml>blablabla<bluh/>bla\nblabla</testxml>\n"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><t>test</t><tt/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<note><to>You</to><from>Me</from><subject>msg</subject><body>This is a test!</body></note>"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml att=\"blah\" y=\"blih\"><t>test</t><tt att=\"test\"/></testxml>"));
        Assert.assertTrue((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"blah<testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"  <testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)" \n\t <testxml><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml <oops><t>test</t><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><badnamespace:t>test</badnamespace:t></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<ns:testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml xmlns:ns=\"http://test.com/test/\"><t>test</t><ns:tt/></ns:testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml badatt=bad><t>test</t></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><t>test</t></><tt badatt=bad/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><t>test</t badatt=\"bad\"></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><t>test</T></><tt/></testxml>"));
        Assert.assertFalse((boolean)XmlUtil.isWellFormedXmlQuickTest((String)"<testxml><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><tt/><t>test</T></><tt/></testxml>"));
    }

    @Test
    public void testFormatXmlFromString1a() {
        String formattedXml = XmlUtil.formatXmlFromString((String)unformattedXml);
        MatcherAssert.assertThat((Object)formattedXml, (Matcher)Matchers.is((Matcher)Matchers.notNullValue()));
        MatcherAssert.assertThat((Object)formattedXml.length(), (Matcher)Matchers.is((Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(unformattedXml.length()))));
        Assert.assertTrue((boolean)formattedXml.contains("\n"));
        LOG.trace("Formatted XML:\n" + formattedXml);
    }

    @Test
    public void testFormatXmlFromString1b() {
        String formattedXml = XmlUtil.formatXmlFromString_alwaysSafe((String)unformattedXml);
        MatcherAssert.assertThat((Object)formattedXml, (Matcher)Matchers.is((Matcher)Matchers.notNullValue()));
        MatcherAssert.assertThat((Object)formattedXml.length(), (Matcher)Matchers.is((Matcher)Matchers.greaterThan((Comparable)Integer.valueOf(unformattedXml.length()))));
        Assert.assertTrue((boolean)formattedXml.contains("\n"));
    }

    @Test(expectedExceptions={Exception.class})
    public void testFormatXmlFromString2a() {
        String formattedXml = XmlUtil.formatXmlFromString((String)badunformattedXml);
    }

    @Test
    public void testFormatXmlFromString2b() {
        String formattedXml = XmlUtil.formatXmlFromString_alwaysSafe((String)badunformattedXml);
        MatcherAssert.assertThat((Object)formattedXml, (Matcher)Matchers.is((Matcher)Matchers.notNullValue()));
        MatcherAssert.assertThat((Object)formattedXml, (Matcher)Matchers.is((Matcher)Matchers.equalTo((Object)badunformattedXml)));
    }

    static {
        PrivateKey pk = null;
        X509Certificate c = null;
        try {
            pk = KeyUtil.pemToAnyPrivateKey((String)testPemPrivateKey, null);
            assert (pk != null);
            c = KeyUtil.pemToX509Certificate((String)testPemCert);
            assert (c != null);
        }
        catch (Exception e) {
            throw new RuntimeException("Static initialize of XmlUtilTest failed: " + e.getMessage(), e);
        }
        testPrivateKey = pk;
        testCert = c;
        testPublicKey = c.getPublicKey();
    }
}

